-- The function 'CreateElectricMultipoleOp()' creates the electric multipole transition operator. 

-- Input: - list of basis orbitals (basisOrbs)
--        - list of radial integrals (RadialIntegrals) for all possible pairs of orbitals connected
--			by the transition operator

-- Output: list of electric multipole transition operators (OpList). Each element in Oplist itself is 
--         a list of length=2. The two entries correspond to the two different contributions in an
--         electric multipole transition. For more details, see PhD thesis (eq. 4.21) of M. Merstorf
--         (https://archiv.ub.uni-heidelberg.de/volltextserver/32091/)  


function CreateElectricMultipoleOp(basisOrbs,RadialIntegrals)

	OpList = {}

	for j = 1, #RadialIntegrals do
					
		-- quantum numbers for orbital b
		n_b = RadialIntegrals[j][1]
		l_b = RadialIntegrals[j][2]
		j_b = RadialIntegrals[j][3]
							
		if j_b == l_b + 1/2 then
			kappa_b = -l_b - 1
		else
			kappa_b = l_b
		end
		
		-- quantum numbers for orbital a
		n_a = RadialIntegrals[j][4]
		l_a = RadialIntegrals[j][5]
		j_a = RadialIntegrals[j][6]
	
		if j_a == l_a + 1/2 then
			kappa_a = -l_a - 1
		else
			kappa_a = l_a
		end
		
		-- multipole order
		J = RadialIntegrals[j][7]
		
		-- initialize transition operator	
		T = 0	
	
		-- loop over all magnetic quantum numbers
		for M = -J, J do
			
			for m_b = -j_b, j_b do
			
				for m_a = -j_a, j_a do
			
					AssignIndices(n_b,l_b,j_b,m_b,n_a,l_a,j_a,m_a,basisOrbs)	
			
					-- Create particle in \tau_b and annihilate in \tau_a
					Op  = NewOperator("Cr",Indices.NF,b_index) * NewOperator("An",Indices.NF,a_index)

					if (l_b+J+l_a)%2 == 0 then
						r       = (-1)^(2*j_b-m_b+1/2) * ThreeJSymbol({j_b,-m_b},{J,M},{j_a,m_a}) * ThreeJSymbol({j_b,-1/2},{j_a,1/2},{J,0}) * Sqrt((2*j_b+1)*(2*j_a+1))
						r_prime = (-1)^(2*j_a-m_a+1/2) * ThreeJSymbol({j_a,-m_a},{J,M},{j_b,m_b}) * ThreeJSymbol({j_a,-1/2},{j_b,1/2},{J,0}) * Sqrt((2*j_b+1)*(2*j_a+1))
					else
						r       = 0
						r_prime = 0
					end
						
					if (n_a == n_b) and (l_a == l_b) and (j_a == j_b) then
						T = T + r * Op 
					else
						T = T + (r * Op - r_prime * ConjugateTranspose(Op))
					end
						
																																															
				end 
			
			end 
		
		end 
	
		-- Add operators to list
		OpList[#OpList+1] = {(kappa_b-kappa_a)/Sqrt(J*(J+1))*Sqrt((2*J+1)/(4*Pi))*T, Sqrt(J*(J+1))*Sqrt((2*J+1)/(4*Pi))*T}
					
	end
	
	return OpList
	
end








